# This python / tkinter program provides a simple example GUI for getting user input 
# and building a custom macro that builds and loops a CSI video frame with minimal blanking.
# Required DLLs for operation are: DPPhyGenCtlRPC.dll and DPhyGenCtlRPCWrapper.dll.
#
# Written by the Moving Pixel Company
from ctypes import *
from ctypes.util import *
from DPhyGenCtlRPCDefines import *
from DPhyGenCtlRPCWrapper import *
from tkinter import *
import os
import time

# Given c_char_p dllMsg that points to a byte array representing string, return the associated Unicode string
# "string_at" gets the byte array from dllMsg
# decode() converts the byte array to a Unicode string
def GetStr(dllMsg):
    return string_at(dllMsg).decode()

def PErr(rc, msg):
    print(msg, " (", rc, ")", sep="")
    return rc

# Assumes global dphyDLL references DPhyGenCtlRPCWrapper DLL
def PrintStatus(callName, rc):
    try:
        # use copy=True to show how to deallocate using Free
        # if copy=False, don't use Free
        if (rc >= 0):
            msg = m_dphyDLL.GetLastStatusMsg(m_client, True)
        else:
            msg = m_dphyDLL.GetLastErrMsg(m_client, True)

        print(callName, "=>", rc, end="")
        print(": \"", GetStr(msg), "\"", sep="")
        m_dphyDLL.Free(m_client, msg)
    except:
        print("Error getting status")

# Assumes global dphyDLL references DPhyGenCtlRPCWrapper DLL
def PrintErrStatus(callName, rc):
    if (rc >= 0): return
    try:
        msg = m_dphyDLL.GetLastErrMsg(m_client, True)
        print(callName, "=>", rc, end="")
        print(": \"", GetStr(msg), "\"", sep="")
    except:
        print("Error getting status")

def PrintResponse(init, resp):
    print(init, end="")
    for i in range(0, len(resp)):
        print (resp[i], "", end="")
    print("");

def GetDUTResponse():
    respIA = POINTER(c_int32)()
    respIALen = c_int32(0)
    rc = m_dphyDLL.RPCQueryRetIA(m_client, RPCCmds.GET_DUT_RESPONSE, byref(respIA), byref(respIALen))
    PrintStatus("GET_DUT_RESPONSE", rc)
    result = [0] * respIALen.value;
    for i in range(len(result)):
        result[i] = respIA[i]
    m_dphyDLL.Free(m_client, respIA)
    PrintResponse("DUT Response = ", result)
    return(rc, result)

def GUIDisconnect():
    global m_GUIConnected
    if (m_GUIConnected):
        rc = m_dphyDLL.Disconnect(m_client)
        m_GUIConnected = False

def GUIConnect(RPCPort):
    global m_GUIConnected
    if (m_GUIConnected):
        InstDisconnect()
    rc = m_dphyDLL.Connect(m_client, "", RPCPort)
    if (rc < 0): return(rc)
    m_GUIConnected = True;
    return(0)

def InstConnect(serNum):
    rc = m_dphyDLL.RPCCmd(m_client, RPCCmds.STOP_PG)
    PrintStatus("STOP_PG", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmd(m_client, RPCCmds.INST_DISCONNECT)
    PrintStatus("INST_DISCONNECT", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmdI(m_client, RPCCmds.INST_CONNECT, serNum)
    PrintStatus("INST_CONNECT", rc)
    if (rc < 0): return(rc)
    connected = m_dphyDLL.RPCCmd(m_client, RPCCmds.IS_INST_CONNECTED)
    PrintStatus("IS_INST_CONNECTED", connected)
    DPhyGenInit()
    return (0 if connected == 1 else connected)

def DPhyGenInit():
    UpdateInstStatus()
    rc = m_dphyDLL.RPCCmdI(m_client, RPCCmds.SET_MIPI_STANDARD, RPCDefs.STD_CSI)
    PrintStatus("SET_MIPI_STANDARD = CSI", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmdII(m_client, RPCCmds.SET_OPTION, RPCDefs.OPT_ENABLE_VIDEO_MODE_IN_MACROS, 0)
    PrintStatus("SET_OPTION (ENABLE_VIDEO_MODE_IN_MACROS = 0)", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmdII(m_client, RPCCmds.SET_OPTION, RPCDefs.OPT_LOOP_COMMANDS, 1)
    PrintStatus("SET_OPTION (LOOP_COMMANDS = 1)", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmdII(m_client, RPCCmds.SET_OPTION, RPCDefs.OPT_ALLOW_IMAGE_RESCALING, 1)
    PrintStatus("SET_OPTION (ALLOW_IMAGE_RESCALING = 1)", rc)
    if (rc < 0): return(rc)

def SendMacro(errCode):
    
    rc = m_dphyDLL.RPCCmdS(m_client, RPCCmds.SEND_MACRO, b"VideoMacro")
    PrintStatus("SEND_MACRO", rc)
    return rc if (errCode == 0) else errCode

def SendVideo(imX, imY, useLSLE, laneCnt, HSBitRate, fn):
    if (not m_GUIConnected):
        Status("No instrument connected.");
        return
    rc = m_dphyDLL.RPCCmdI(m_client, RPCCmds.SET_LANE_CNT, laneCnt)
    PrintStatus("SET_LANE_CNT", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmdF(m_client, RPCCmds.SET_HS_BIT_RATE, HSBitRate)
    PrintStatus("SET_HS_BIT_RATE", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmdI(m_client, RPCCmds.SET_TIMING_ENABLE_CSI_LSLE_MODE, useLSLE)
    PrintStatus("SET_TIMING_ENABLE_CSI_LSLE_MODE", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmdI(m_client, RPCCmds.SET_TIMING_HACTIVE, imX)
    PrintStatus("SET_TIMING_HACTIVE", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmdI(m_client, RPCCmds.SET_TIMING_VACTIVE, imY)
    PrintStatus("SET_TIMING_VACTIVE", rc)
    if (rc < 0): return(rc)
    rc = m_dphyDLL.RPCCmdIIS(m_client, RPCCmds.LOAD_FRAME, 0, RPCDefs.PIXEL_STREAM_RGB888, c_char_p(fn.encode('utf-8'))) 
    PrintStatus("LOAD_FRAME", rc)
    if (rc < 0): return(rc)

    rc = m_dphyDLL.RPCCmd(m_client, RPCCmds.START_MACRO)
    PrintStatus("START_MACRO", rc)
    if (rc < 0): return(SendMacro())
    rc = m_dphyDLL.MIPICmd(m_client, RPCDefs.FRAME_START, 0, False, 0, 0, 0, 0, 0, b"", None, 0)
    PrintStatus("FRAME_START", rc)
    if (rc < 0): return(SendMacro())
    for y in range(imY):
        if (useLSLE):
            rc = m_dphyDLL.MIPICmd(m_client, RPCDefs.LINE_START, 0, False, 0, 0, y, 0, 0, b"", None, 0)
            PrintErrStatus("LINE_START", rc)
            if (rc < 0): return(SendMacro())
        rc = m_dphyDLL.MIPICmd(m_client, RPCDefs.PIXEL_STREAM_RGB888, 0, False, 0, 0, 3*imX*y, 3*imX, 0, b"USERFRAME0", None, 0)
        PrintErrStatus("PIXEL_STREAM_RGB888", rc)
        if (useLSLE):
            rc = m_dphyDLL.MIPICmd(m_client, RPCDefs.LINE_END, 0, False, 0, 0, y, 0, 0, b"", None, 0)
            PrintErrStatus("LINE_END", rc)
            if (rc < 0): return(SendMacro())
    rc = m_dphyDLL.MIPICmd(m_client, RPCDefs.FRAME_END, 0, False, 0, 0, 0, 0, 0, b"", None, 0)
    PrintStatus("FRAME_END", rc)
    if (rc < 0): return(SendMacro())
    SendMacro(0)

def IsProgramRunning():
    if (not m_GUIConnected):
        return 0
    rc = m_dphyDLL.RPCCmd(m_client, RPCCmds.IS_PROGRAM_RUNNING)
    return(rc)

def TestCommands():
    m_dphyDLL.RPCCmd(m_client, RPCCmds.START_EDIT_CONFIG)
    rc = m_dphyDLL.RPCCmdF(m_client, RPCCmds.SET_BTA_WAIT_TIME, 110e-6)
    PrintStatus("SET_BTA_WAIT_TIME", rc)
    m_dphyDLL.RPCCmd(m_client, RPCCmds.END_EDIT_CONFIG)
    rc = m_dphyDLL.RPCCmdS(m_client, RPCCmds.SAVE_TIMING_CONFIG, b"NewConfig")
    PrintStatus("SAVE_TIMING_CONFIG", rc)
    rc = m_dphyDLL.RPCCmdS(m_client, RPCCmds.SELECT_TIMING_CONFIG, b"640x480p")
    PrintStatus("SELECT_TIMING_CONFIG", rc)
    rc = m_dphyDLL.MIPICmd(m_client, RPCDefs.PACKED_PIXEL_STREAM_888, 0, False, 0, 0, 1, 0, 0, b"c:\\jobs\\test.bmp", None, 0)
    PrintStatus("PIXEL_STREAM_RGB888", rc)
    rc = m_dphyDLL.RPCCmd(m_client, RPCCmds.IS_PROGRAM_RUNNING)
    PrintStatus("IS_PROGRAM_RUNNING", rc)
    rc = m_dphyDLL.RPCCmd(m_client, RPCCmds.IS_INST_CONNECTED)
    PrintStatus("IS_INST_CONNECTED", rc)
    rc = m_dphyDLL.MIPICmd(m_client, RPCDefs.DSI_BLANKING_PKT, 0, False, 0, 0, 100, 0, 0, b"", None, 0)
    PrintStatus("DSI_BLANKING_PKT", rc)
    rc = m_dphyDLL.RPCCmdI(m_client, RPCCmds.SET_LANE_CNT, 2)
    PrintStatus("SET_LANE_CNT", rc)
    rc = m_dphyDLL.RPCCmdF(m_client, RPCCmds.SET_BTA_WAIT_TIME, 100e-6)
    PrintStatus("SET_BTA_WAIT_TIME", rc)
    rc, resp = GetDUTResponse()

##################################################
# TKInter support

class CheckBox(Frame):
    def __init__(self, master, text, callback = None):
        Frame.__init__(self, master)
        self.checked = IntVar()
        self.cb = Checkbutton(master, text = text, variable = self.checked)
        if (callback != None): 
            self.checkbox["command"] = callback

class TextBox(Frame):
    def __init__(self, master, width = 10, text = "", callback = None):
        Frame.__init__(self, master)
        self.text = StringVar()
        self.text.set(text)
        self.tb = Entry(master, textvariable = self.text)
        self.tb["width"] = width
        if (callback != None): 
            self.tb["validate"] = "focusout"
            self.tb["validatecommand"] = callback

class DropDown(Frame):
    def __init__(self, master, options, callback = None):
        Frame.__init__(self, master)
        self.option = StringVar()
        self.option.set(options[0])
        self.dd = OptionMenu(master, self.option, *options)
        if (callback != None): 
            self.option.trace("w", callback)

def AddLabel(text, row, col):
    lbl = Label(Interior)
    lbl["text"] = text
    lbl.grid(row = row, column = col, sticky = W, padx = 5)
    return(lbl)

def AddTB(text, width, row, col, val = "", callback = None):
    lbl = Label(Interior)
    lbl["text"] = text
    lbl.grid(row = row, column = col, sticky = E, padx = 5)
    textbox = TextBox(Interior, width, val, callback)
    textbox.tb.grid(row = row, column = col+1, sticky = W, pady = 2)
    return(textbox)

def AddCheckBox(text, row, col, padx = 0, pady = 0, callback = None):
    checkbox = CheckBox(Interior, text = text, callback = callback)
    checkbox.cb.grid(row = row, column = col, padx = padx, pady = pady, sticky = W)
    return(checkbox)

def AddBut(text, row, col, cb, padx = 0, pady = 0, sizepadx = 5, sizepady = 0, width = -1):
    but = Button(Interior, text = text, command = cb, padx = sizepadx, pady = sizepady)
    if (width != -1):
        but.config(width = width)
    but.grid(row=row, column = col, padx = padx, pady = pady)
    return(but)

def AddDropDown(text, row, col, options, callback = None):
    lbl = Label(Interior)
    lbl["text"] = text
    lbl.grid(row = row, column = col, sticky = E, padx = 5)
    dropdown = DropDown(Interior, options, callback)
    dropdown.dd.grid(row = row, column = col + 1, sticky = W, ipadx = 5, pady = 5)
    return(dropdown)

def StatusEvent():
    UpdateInstStatus()
    root.after(1000, StatusEvent)

def Status(text):
    StatusBar["text"] = text

def UpdateInstStatus():
    if (m_GUIConnected):
        ConnStatusLbl["text"] = "Connection: Active"
    else:
        ConnStatusLbl["text"] = "Connection: Inactive"

    if (IsProgramRunning() == 1):
        PGStatusLbl["text"] = "PG: Running"
    else:
        PGStatusLbl["text"] = "PG: Idle"

def BrowseButtonClick():
    from tkinter import filedialog
    curdir = os.getcwd()
    fn = filedialog.askopenfilename(title = "Image File Select", initialdir = curdir)
    FileTB.text.set(fn)

def ConnectButtonClick():
    if (not m_GUIConnected): 
        try:
            portNum = int(PortTB.tb.get())
        except ValueError:
            Status("Error: can't parse port as integer: " + PortTB.get())
            return
        rc = GUIConnect(portNum)
        if (rc == 0):
            Status("Connected successfully.")
            ConnectBut["text"] = "Disconnect"
            DPhyGenInit()
        else:
            Status("Connection failed.")
    else:
        GUIDisconnect();
        ConnectBut["text"] = "Connect"
        Status("Disconnected.")

# Builds macro for CSI video frame with minimal blanking and loops it
def Send():
    HSBitRate = SToF("HS Bit Rate", HSBitRateTB.text.get())
    if (HSBitRate == None): return
    HSBitRate *= 1e6
    HAct = SToI("HActive", HActTB.text.get())
    if (HAct == None): return
    VAct = SToI("VActive", VActTB.text.get())
    if (VAct == None): return
    laneCnt = SToI("Lane Cnt", LaneCntDD.option.get())
    if (laneCnt == None): return
    useLSLE = UseLSLECB.checked.get()
    fn = FileTB.text.get()
    SendVideo(HAct, VAct, useLSLE, laneCnt, HSBitRate, fn)

def Exit():
    sys.exit()

def SToI(varName, valS):
    try:
        valI = int(valS)
        return(valI)
    except:
        Status("Can't parse " + varName + " as integer: " + valS)

def SToF(varName, valS):
    try:
        valF = float(valS)
        return(valF)
    except:
        Status("Can't parse " + varName + " as floating point: " + valS)

root = Tk()
root.resizable(width = False, height = False)
root.title("Demo Python Controller")

# define main window frame 
StatusBar = Label(root, bd = 1, relief = SUNKEN, anchor = W)
StatusBar.pack(side = BOTTOM, fill = X)
Interior = Frame(root)
Interior.pack(side = TOP, fill = BOTH, expand = True)
Interior["padx"] = 30
Interior["pady"] = 6

# add main window controls
row = 0
col = 0
PortTB = AddTB("Port", 8, row, col, 2799)
ConnectBut = AddBut("Connect", row, col+2, ConnectButtonClick, 10, 20, 5, 0, 12)
HSBitRateTB = AddTB("HS Bit Rate (Mbps)", 8, row + 1, col, 100)
UseLSLECB = AddCheckBox("Use LS/LE", row + 1, col + 2, 5)
HActTB = AddTB("HActive (Pix)", 8, row+2, col, 640)
VActTB = AddTB("VActive (Line)", 8, row+3, col, 480)
ConnStatusLbl = AddLabel("", row+2, col+2)
PGStatusLbl = AddLabel("", row+3, col+2)
LaneCntDD = AddDropDown("Lane Cnt", row + 4, col, ["1", "2", "3", "4"])
FileTB = AddTB("Image File", 40, row+5, col, "")
FileTB.tb.grid(columnspan = 2, sticky=W+E)
BrowseBut = AddBut("...", row+5, col+3, BrowseButtonClick, 5)
SendBut = AddBut("Send", row+6, col + 1, Send,  0, 20, 20, 12)
ExitBut = AddBut("Exit", row+6, col + 2, Exit,  0, 20, 20, 12)

# initialize DLL and client global references
m_dphyDLL = OpenDPhyGenCtlRPCWrapper()
m_client = m_dphyDLL.NewClient()
m_GUIConnected = False

StatusEvent()

# start GUI
root.mainloop()













