// This code defines an unmanaged wrapper DLL for the managed .NET DLL DPhyGenCtlRPC
// provided by the Moving Pixel Company to remotely control the DPhy Generator (P344) 
// instrument.  This wrapper DLL can be easily used by languages that do not directly 
// interface with the .NET environment such as Python, Matlab, Labview, etc.
//
// Example C++ usage is included in the TestWrapper project, included in this solution.
// In addition, there is an example Python project (DPhyGenPythonCtl) that demonstrates
// interfacing to this DLL as well.  Note that the user program must have both this
// DLL (DPhyGenCtlRPCWrapper.dll) as well as DPhyGenCtlRPC.dll present for correct 
// operation.  Also note that while DPhyGenCtlRPC.dll will work for both x86 and x64
// architectures, this project builds only an x86 version of DPhyGenCtlRPCWrapp.dll. 
// Some languages cannot support the "Any-CPU' DLL architecture and require an explicit
// x86 or x64 version.  The user can, of course, modify this project to build an x64
// version of the DLL if desired.
//
// Basically, the initial steps user code should take to use the DLL are:
//
// 1) Intantiate a "client" object using the NewClient DLL method.  The client object
// is to be provided in all subsequent DLL calls as the first argument.
//
// 2) Call the Connect DLL method, giving the hostname and IP port to 
// use to establish a connection to the DPhyGenCtl software.  DPhyGenCtl must
// have enabled the RPC port using the Connect->Enable RPC menu option prior to
// this call.
//
// After these two steps, RPC calls can be made to DPhyGenCtl using the RPCCmd and
// RPCQuery calls, which have various suffixes to denote the argument and/or return
// types.  'I' implies an integer argument, 'S' implies a string argument, and 'F'
// represents a float argument.  For example, RPCCmdII has two integer arguments.
// To send a MIPI command,  use the MIPICmd DLL call.
//
// Note this code is provided to the user "as-is" and is not actively supported
// by the Moving Pixel Company.  Please see "readme.txt" in the toplevel directory
// for more information about the use of this library and example projects.

#include "stdio.h" 
#include "stdlib.h"
#include <msclr\auto_gcroot.h>
#include <new>
#include "DPhyGenCtlRPCWrapper.h"

using namespace System;
using namespace System::IO;
using namespace System::Reflection;
using namespace System::Runtime::InteropServices;
using namespace System::Text;
using namespace DPhyGenCtlRPC;

#define CLIENT m_privateWrapper->m_client
#define ERRMSG m_privateWrapper->m_msgs->m_errMsg
#define STATUSMSG m_privateWrapper->m_msgs->m_statusMsg

#define PCLIENT reinterpret_cast<DPhyGenCtlRPCWrapper*>(client)

static Assembly^ AssemblyResolve(Object^ Sender, ResolveEventArgs^ args)
{
	AssemblyName^ assemblyName = gcnew AssemblyName(args->Name);

	if (assemblyName->Name == "DPhyGenCtlRPC")
	{
		String^ path = Path::Combine(Path::GetDirectoryName(Assembly::GetExecutingAssembly()->Location), "DPhyGenCtlRPC.dll");
		return Assembly::LoadFile(path);
	}

	return nullptr;
}

//////////////////////////////////////////////////
// Managed/Unmanaged string conversion routines

static unsigned char *ConvertBAToUnmanaged(array<Byte>^ ba)
{
	if (ba == nullptr)
		return NULL;
	unsigned char* buf = (unsigned char *)malloc(ba->Length);
	System::Runtime::InteropServices::Marshal::Copy(ba, 0, IntPtr((void *)buf), ba->Length);
	return(buf);
}

static Int32 *ConvertIAToUnmanaged(array<Int32>^ ia)
{
	if (ia == nullptr)
		return NULL;
	Int32* buf = (Int32 *)malloc(4 * ia->Length);
	System::Runtime::InteropServices::Marshal::Copy(ia, 0, IntPtr((void *)buf), ia->Length);
	return(buf);
}

static char *ConvertStringToUnmanaged(String^ str)
{
	if (str == nullptr)
		return NULL;
	array<Byte> ^ba = System::Text::Encoding::UTF8->GetBytes(str);
	char* buf = (char *)malloc(ba->Length + 1);
	System::Runtime::InteropServices::Marshal::Copy(ba, 0, IntPtr((void *)buf), ba->Length);
	buf[ba->Length] = 0;

//	char* buf = (char*)(void *)System::Runtime::InteropServices::Marshal::StringToHGlobalAnsi(str);
	return(buf);
}

static array<Byte> ^ConvertBAToManaged(unsigned char* buf, int len)
{
	if (buf == NULL)
		return nullptr;
	array<Byte>^ ba = gcnew array<Byte>(len);
	System::Runtime::InteropServices::Marshal::Copy(IntPtr((void *)buf), ba, 0, len);
	return(ba);
}

static String ^ConvertStringToManaged(char* str)
{
	if (str == NULL)
		return nullptr;
	return(gcnew String(str));
}

//////////////////////////////////////////////////
// Managed class to hold error and status messages
// (Unmanaged classes cannot hold static references to managed strings)
// I used this class because I couldn't get auto_gcroot String^ references 
// to work properly.
ref class Msgs
{
public:
	String^ m_errMsg;
	String^ m_statusMsg;
};

//////////////////////////////////////////////////
//  Unmanaged class to hold auto_gcroot variables
// auto_gcroot is a method to hold reference to managed classes
class DPhyGenCtlRPCPrivateWrapper
{
    public: 
		msclr::auto_gcroot<DPhyGenCtlRPCClient^> m_client;
		msclr::auto_gcroot<Msgs^> m_msgs;
};

//////////////////////////////////////////////////
//  CLI/C++ class to provide client interface to DLL

DPhyGenCtlRPCWrapper::DPhyGenCtlRPCWrapper()
{
	m_privateWrapper = new DPhyGenCtlRPCPrivateWrapper();
	m_privateWrapper->m_client = gcnew DPhyGenCtlRPCClient();
	m_privateWrapper->m_msgs = gcnew Msgs();
	m_lastErrMsg = (char *)malloc(1);
	m_lastErrMsg[0] = 0;
	m_lastStatusMsg = (char *)malloc(1);
	m_lastStatusMsg[0] = 0;
}

DPhyGenCtlRPCWrapper::~DPhyGenCtlRPCWrapper()
{
	delete m_privateWrapper;
	free (m_lastErrMsg);
	free(m_lastStatusMsg);
}

int DPhyGenCtlRPCWrapper::Connect(const char* hostName, int port)
{
	return CLIENT->Connect(gcnew System::String(hostName), port);
}

void DPhyGenCtlRPCWrapper::Disconnect()
{
	CLIENT->Disconnect();
}

char* DPhyGenCtlRPCWrapper::GetLastErrMsg(bool copy)
{
	if (copy) {
		char* msg = (char*)malloc(strlen(m_lastErrMsg) + 1);
		strcpy(msg, m_lastErrMsg);
		return(msg);
	}
	else
		return(m_lastErrMsg);
}

char* DPhyGenCtlRPCWrapper::GetLastStatusMsg(bool copy)
{
	if (copy) {
		char* msg = (char*)malloc(strlen(m_lastStatusMsg) + 1);
		strcpy(msg, m_lastStatusMsg);
		return(msg);
	}
	else
		return(m_lastStatusMsg);
}

void DPhyGenCtlRPCWrapper::Free(void* ptr)
{
	if (ptr != NULL)
		free(ptr);
}

int DPhyGenCtlRPCWrapper::UpdateMsgs(int rc)
{
	free(m_lastErrMsg);
	free(m_lastStatusMsg);
	m_lastErrMsg = ConvertStringToUnmanaged(ERRMSG);
	m_lastStatusMsg = ConvertStringToUnmanaged(STATUSMSG);
	return(rc);
}

int DPhyGenCtlRPCWrapper::RPCCmd(int cmdCode)
{
	int rc = CLIENT->RPCCmd(cmdCode, ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdI(int cmdCode, int arg1)
{
	int rc = CLIENT->RPCCmdI(cmdCode, arg1, ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdF(int cmdCode, float arg1)
{
	int rc = CLIENT->RPCCmdF(cmdCode, arg1, ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdS(int cmdCode, const char* arg1)
{
	int rc = CLIENT->RPCCmdS(cmdCode, gcnew String(arg1), ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdII(int cmdCode, int arg1, int arg2)
{
	int rc = CLIENT->RPCCmdII(cmdCode, arg1, arg2, ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdIF(int cmdCode, int arg1, float arg2)
{
	int rc = CLIENT->RPCCmdIF(cmdCode, arg1, arg2, ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdIS(int cmdCode, int arg1, const char* arg2)
{
	int rc = CLIENT->RPCCmdIS(cmdCode, arg1, gcnew String(arg2), ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdIII(int cmdCode, int arg1, int arg2, int arg3)
{
	int rc = CLIENT->RPCCmdIII(cmdCode, arg1, arg2, arg3, ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdIIS(int cmdCode, int arg1, int arg2, const char* arg3)
{
	int rc = CLIENT->RPCCmdIIS(cmdCode, arg1, arg2, gcnew String(arg3), ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdSIS(int cmdCode, const char* arg1, int arg2, const char* arg3)
{
	int rc = CLIENT->RPCCmdSIS(cmdCode, gcnew String(arg1), arg2, gcnew String(arg3), ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdIIBA(int cmdCode, int arg1, int arg2, unsigned char* arg3, int arg3Len)
{
	array<Byte>^ ba = ConvertBAToManaged(arg3, arg3Len);
	int rc = CLIENT->RPCCmdIIBA(cmdCode, arg1, arg2, ba, ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdIISS(int cmdCode, int arg1, int arg2, const char* arg3, const char* arg4)
{
	int rc = CLIENT->RPCCmdIISS(cmdCode, arg1, arg2, gcnew String(arg3), gcnew String(arg4), ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCCmdSISI(int cmdCode, const char* arg1, int arg2, const char* arg3, int arg4)
{
	int rc = CLIENT->RPCCmdSISI(cmdCode, gcnew String(arg1), arg2, gcnew String(arg3), arg4, ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCQueryIRetS(int cmdCode, int arg1, char** respVal)
{
	String^ respStr;
	int rc = CLIENT->RPCQueryIRetS(cmdCode, arg1, respStr, ERRMSG, STATUSMSG);
	*respVal = ConvertStringToUnmanaged(respStr);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCQuerySRetS(int cmdCode, const char* arg1, char** respVal)
{
	String^ respStr;
	int rc = CLIENT->RPCQuerySRetS(cmdCode, gcnew String(arg1), respStr, ERRMSG, STATUSMSG);
	*respVal = ConvertStringToUnmanaged(respStr);
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCQueryRetBA(int cmdCode, unsigned char** respVal, int* respValLen)
{
	array<Byte> ^ba;
	int rc = CLIENT->RPCQueryRetBA(cmdCode, ba, ERRMSG, STATUSMSG);
	*respVal = ConvertBAToUnmanaged(ba);
	*respValLen = ba->Length;
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::RPCQueryRetIA(int cmdCode, int** respVal, int* respValLen)
{
	array<Int32> ^ia = gcnew array<Int32>(0);
	int rc = CLIENT->RPCQueryRetIA(cmdCode, ia, ERRMSG, STATUSMSG);
	*respVal = ConvertIAToUnmanaged(ia);
	*respValLen = ia->Length;
	return(UpdateMsgs(rc));
}

int DPhyGenCtlRPCWrapper::MIPICmd(int cmdCode, int DCSCmdCode, bool BTA, int DTMode,
	int VC, int arg1, int arg2, int arg3, const char* fn, unsigned char* data, int dataLen)
{
	array<Byte> ^ba = ConvertBAToManaged(data, dataLen);
	int rc = CLIENT->MIPICmd(cmdCode, DCSCmdCode, BTA, DTMode, VC,
		arg1, arg2, arg3, gcnew String(fn), ba, ERRMSG, STATUSMSG);
	return(UpdateMsgs(rc));
}

//////////////////////////////////////////////////
//  DLL interface routines to call wrapper functions

extern "C"
{
	bool isInitialized = false;

	void Initialize()
	{
		if (!isInitialized)
		{
			AppDomain::CurrentDomain->AssemblyResolve += gcnew ResolveEventHandler(AssemblyResolve);
			isInitialized = true;
		}
	}

	__declspec(dllexport) void* NewClient()
	{
		Initialize();
		return new(std::nothrow)DPhyGenCtlRPCWrapper();
	}

	__declspec(dllexport) void DeleteClient(void* client)
	{
		delete client;
	}

	__declspec(dllexport) char* GetLastErrMsg(void* client, bool copy)
	{
		return PCLIENT->GetLastErrMsg(copy);
	}

	__declspec(dllexport) char* GetLastStatusMsg(void* client, bool copy)
	{
		return PCLIENT->GetLastStatusMsg(copy);
	}

	__declspec(dllexport) void Free(void* client, void* ptr)
	{
		PCLIENT->Free(ptr);
	}

	__declspec(dllexport) int Connect(void* client, const char* hostname, int port)
	{
		return PCLIENT->Connect(hostname, port);
	}

	__declspec(dllexport) void Disconnect(void* client)
	{
		return PCLIENT->Disconnect();
	}

	__declspec(dllexport) int RPCCmd(void* client, int cmdCode)
	{
		return PCLIENT->RPCCmd(cmdCode);
	}

	__declspec(dllexport) int RPCCmdI(void* client, int cmdCode, int arg1)
	{
		return PCLIENT->RPCCmdI(cmdCode, arg1);
	}

	__declspec(dllexport) int RPCCmdF(void* client, int cmdCode, float arg1)
	{
		return PCLIENT->RPCCmdF(cmdCode, arg1);
	}

	__declspec(dllexport) int RPCCmdS(void* client, int cmdCode, char* arg1)
	{
		return PCLIENT->RPCCmdS(cmdCode, arg1);
	}

	__declspec(dllexport) int RPCCmdII(void* client, int cmdCode, int arg1, int arg2)
	{
		return PCLIENT->RPCCmdII(cmdCode, arg1, arg2);
	}

	__declspec(dllexport) int RPCCmdIF(void* client, int cmdCode, int arg1, float arg2)
	{
		return PCLIENT->RPCCmdIF(cmdCode, arg1, arg2);
	}

	__declspec(dllexport) int RPCCmdIS(void* client, int cmdCode, int arg1, const char* arg2)
	{
		return PCLIENT->RPCCmdIS(cmdCode, arg1, arg2);
	}

	__declspec(dllexport) int RPCCmdIII(void* client, int cmdCode, int arg1, int arg2, int arg3)
	{
		return PCLIENT->RPCCmdIII(cmdCode, arg1, arg2, arg3);
	}

	__declspec(dllexport) int RPCCmdIIS(void* client, int cmdCode, int arg1, int arg2, const char* arg3)
	{
		return PCLIENT->RPCCmdIIS(cmdCode, arg1, arg2, arg3);
	}

	__declspec(dllexport) int RPCCmdSIS(void* client, int cmdCode, const char* arg1, int arg2, const char* arg3)
	{
		return PCLIENT->RPCCmdSIS(cmdCode, arg1, arg2, arg3);
	}

	__declspec(dllexport) int RPCCmdIIBA(void* client, int cmdCode, int arg1, int arg2, unsigned char* arg3, int arg3Len)
	{
		return PCLIENT->RPCCmdIIBA(cmdCode, arg1, arg2, arg3, arg3Len);
	}

	__declspec(dllexport) int RPCCmdIISS(void* client, int cmdCode, int arg1, int arg2, const char* arg3, const char* arg4)
	{
		return PCLIENT->RPCCmdIISS(cmdCode, arg1, arg2, arg3, arg4);
	}

	__declspec(dllexport) int RPCCmdSISI(void* client, int cmdCode, const char* arg1, int arg2, const char* arg3, int arg4)
	{
		return PCLIENT->RPCCmdSISI(cmdCode, arg1, arg2, arg3, arg4);
	}

	__declspec(dllexport) int RPCQueryIRetS(void* client, int cmdCode, int arg1, char** respVal)
	{
		return PCLIENT->RPCQueryIRetS(cmdCode, arg1, respVal);
	}

	__declspec(dllexport) int RPCQuerySRetS(void* client, int cmdCode, const char* arg1, char** respVal)
	{
		return PCLIENT->RPCQuerySRetS(cmdCode, arg1, respVal);
	}

	__declspec(dllexport) int RPCQueryRetBA(void* client, int cmdCode, unsigned char** respVal, int* respValLen)
	{
		return PCLIENT->RPCQueryRetBA(cmdCode, respVal, respValLen);
	}

	__declspec(dllexport) int RPCQueryRetIA(void* client, int cmdCode, int** respVal, int* respValLen)
	{
		return PCLIENT->RPCQueryRetIA(cmdCode, respVal, respValLen);
	}

	__declspec(dllexport) int MIPICmd(void* client, int cmdCode, int DCSCmdCode, bool BTA, int DTMode,
		                              int VC, int arg1, int arg2, int arg3, const char* fn,
		                              unsigned char* data, int dataLen)
	{
		return PCLIENT->MIPICmd(cmdCode, DCSCmdCode, BTA, DTMode,
			VC, arg1, arg2, arg3, fn, data, dataLen);
	}
}

