﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.IO;
using System.Windows.Forms;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using DPhyGenCtlRPC;

namespace CSIVideoPlayer
{
    public partial class CSIVideoPlayerForm : Form
    {
        // configuration file version
        public const int CONFIG_FILE_VERSION = 1;

        // IP port to use to connect to DPhyGenCtl
        public const int SERVER_PORT = 2799;

        // user and derived settings for frame construction
        public Settings m_set = new Settings();

        // instance of DPhYGenCtlRPCClient to communicate with DPhyGenCtl
        public DPhyGenCtlRPCClient m_client = new DPhyGenCtlRPCClient();

        // flag to prevent event handling recursion during the setting of control data
        public bool m_lock = false;

        // flag indicating we are connected to DPhyGenCtl
        public bool m_connected = false;

        // flag indicating the HSFreq and/or LaneCnt has changed
        // and we need to measure the clock on/off and short packet times
        public bool m_needMeasure = true;

        //////////////////////////////////////////////////////////////////////////////////////////
        // Form support routines
        //////////////////////////////////////////////////////////////////////////////////////////

        // Routine to clear status and hold for a short period of time so "blink" is seen
        // if it is immediately rewritten.
        public void ClearStatus()
        {
            Status("");
            System.Threading.Thread.Sleep(100);
        }

        // Display status message on status bar
        public void Status(string txt)
        {
            statusLabel.Text = txt;
            Application.DoEvents();
        }

        // print error message and return error code
        public int PE(int rc, string eMsg)
        {
            Status(eMsg);
            return (rc);
        }

        // print error message if no error message has yet been shown
        public int DefaultPE(int rc, string eMsg)
        {
            if (statusLabel.Text == "")
                Status(string.Format("Error {0} (rc = {1})", eMsg, rc));
            return (rc);
        }

        // Parse an integer value from the contents of the given text box
        // If parsing fails, display an error in the status bar using errStr as the field name.
        // If parsing succeeds, return parsed value in val.
        public void ParseIntTextBox(TextBox tb, string errStr, ref int val)
        {
            int parsedVal = 0;
            if (!Int32.TryParse(tb.Text, out parsedVal))
            {
                Status("Can't parse " + errStr + " as integer.");
                tb.Text = val.ToString();
            }
            else
                val = parsedVal;
        }

        // Parse a floating point value from the contents of the given text box
        // If parsing fails, display an error in the status bar using errStr as the field name.
        // If parsing succeeds, return parsed value in val.
        public void ParseFloatTextBox(TextBox tb, string errStr, float scale, ref float val)
        {
            float parsedVal = 0;
            if (!Single.TryParse(tb.Text, out parsedVal))
            {
                Status("Can't parse " + errStr + " as floating point value.");
                tb.Text = (val / scale).ToString();
            }
            else
                val = parsedVal * scale;
        }

        // Browse for an image file starting with the directory contained in the given TextBox
        // If successful, display the user-selected filename in the Textbox
        public void BrowseForImageFile(TextBox tb)
        {
            // Create an OpenFileDialog object.
            OpenFileDialog dlg = new OpenFileDialog();
            dlg.Filter = "BMP files|*.bmp";
            dlg.CheckFileExists = true;

            // try settting the initial directory to the path in the TextBox
            try
            {
                dlg.InitialDirectory = Path.GetDirectoryName(tb.Text);
            }
            catch
            {
            }

            // show the file dialog, storing the selected file in the textbox if successful
            if (dlg.ShowDialog() == DialogResult.OK)
                tb.Text = dlg.FileName;
        }

        // enable or disable all the Frame2 parameter controls
        public void EnableFrame2Controls(bool enable)
        {
            VCCombo2.Enabled = enable;
            FormatCombo2.Enabled = enable;
            UseLSLECheckBox2.Enabled = enable;
            ImageFileTextBox2.Enabled = enable;
            HActiveTextBox2.Enabled = enable;
            HBlankTextBox2.Enabled = enable;
            VActiveTextBox2.Enabled = enable;
            VBlankTextBox2.Enabled = enable;
        }

        //////////////////////////////////////////////////////////////////////////////////////////
        // Form handlers
        //////////////////////////////////////////////////////////////////////////////////////////

        // constructor for the CSIVideoPlayerForm
        public CSIVideoPlayerForm()
        {
            InitializeComponent();
        }

        // initialize the main form at start up
        public void CSIVideoPlayerForm_Load(object sender, EventArgs e)
        {
            // set form colors
            this.BackColor = Color.FromArgb(100, 130, 120);
            this.ForeColor = Color.White;

            // initialize button colors
            double scale = 0.8;
            int R = (int)(BackColor.R * scale);
            int G = (int)(BackColor.G * scale);
            int B = (int)(BackColor.B * scale);
            Color buttonBackColor = Color.FromArgb(R, G, B);
            Button[] bts = new Button[] { BrowseButton1, BrowseButton2, ConnectButton, SendButton, 
                                          MinHBlankButton1, MinHBlankButton2 };
            foreach (Button bt in bts)
                bt.BackColor = buttonBackColor;

            // set menu background colors
            Color menuBackColor = Color.FromArgb(200, 210, 200);
            MainMenu.BackColor = buttonBackColor;
            fileToolStripMenuItem.BackColor = menuBackColor;
            aboutToolStripMenuItem.BackColor = menuBackColor;

            Color menuForeColor = Color.Black;
            MainMenu.ForeColor = menuForeColor;

            // initialize event handlers text box event handlers
            TextBox[] tbs = new TextBox[] {HActiveTextBox1, HBlankTextBox1, VActiveTextBox1, VBlankTextBox1,
                                           HActiveTextBox2, HBlankTextBox2, VActiveTextBox2, VBlankTextBox2,
                                           ImageFileTextBox1, ImageFileTextBox2, PortNumTextBox, HSFreqTextBox};
            foreach (TextBox tb in tbs)
            {
                tb.Leave += new EventHandler(TextBox_Leave);
                tb.KeyDown += new KeyEventHandler(TextBox_KeyDown);
            }

            // fill in format combo boxes
            for (int i = 0; i < CSI.Formats.Length; i++)
            {
                FormatCombo1.Items.Add(CSI.CSIVideoCmdString(CSI.Formats[i]));
                FormatCombo2.Items.Add(CSI.CSIVideoCmdString(CSI.Formats[i]));
            }

            // fill in sequence format combo box
            // (strings match SEQ_x defines)
            m_lock = true;
            SeqFormatCombo.Items.Clear();
            SeqFormatCombo.Items.Add("Single Frame");
            SeqFormatCombo.Items.Add("Sequential Frames");
            SeqFormatCombo.Items.Add("Concentric Sequential Frames");
            SeqFormatCombo.Items.Add("Concentric Interleaved Frames");
            m_lock = false;

            // initialize settins in controls
            SetControlData();
        }

        public void CSIVideoPlayerForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            // disconnect on exit
            if (m_connected)
                ConnectButton_Click(null, null);
        }

        // set the form controls from the settings in m_set
        public void SetControlData()
        {
            // use m_lock to prevent recursion when we set control values
            // (which might trigger event that calls SetControlData again)
            if (m_lock) return;
            m_lock = true;

            PortNumTextBox.Text = m_set.port.ToString();

            VCCombo1.SelectedIndex = m_set.VC[0];
            VCCombo2.SelectedIndex = m_set.VC[1];

            int inx = Array.IndexOf(CSI.Formats, m_set.videoFormat[0]);
            if (inx < 0) inx = 0;
            FormatCombo1.SelectedIndex = inx;
            inx = Array.IndexOf(CSI.Formats, m_set.videoFormat[1]);
            if (inx < 0) inx = 0;
            FormatCombo2.SelectedIndex = inx;

            UseLSLECheckBox1.Checked = m_set.UseLSLE[0];
            UseLSLECheckBox2.Checked = m_set.UseLSLE[1];

            ImageFileTextBox1.Text = m_set.imageFn[0];
            ImageFileTextBox2.Text = m_set.imageFn[1];

            HActiveTextBox1.Text = m_set.hact[0].ToString();
            HActiveTextBox2.Text = m_set.hact[1].ToString();

            HBlankTextBox1.Text = m_set.hblank[0].ToString();
            HBlankTextBox2.Text = m_set.hblank[1].ToString();

            VActiveTextBox1.Text = m_set.vact[0].ToString();
            VActiveTextBox2.Text = m_set.vact[1].ToString();

            VBlankTextBox1.Text = m_set.vblank[0].ToString();
            VBlankTextBox2.Text = m_set.vblank[1].ToString();

            SeqFormatCombo.SelectedIndex = m_set.seqFormat;
            EqualLinesCheckBox.Checked = m_set.equalLines;
            SeparateFSFECheckBox.Checked = m_set.separateFSFE;
            QuantizeCheckBox.Checked = m_set.quantLineLength;
            ClockOffDuringBlankingCheckBox.Checked = m_set.clockOffDuringBlanking;
            LoopCheckBox.Checked = m_set.loopFrames;

            EnableFrame2Controls(m_set.seqFormat != Settings.SEQ_SINGLE_FRAME);

            HSFreqTextBox.Text = (m_set.hsFreq / 1e+6).ToString("F1");
            LaneCntCombo.SelectedIndex = m_set.laneCnt - 1;

            m_lock = false;
        }

        // retrieve settings in control into m_set
        public void GetControlData()
        {
            m_set.videoFormat[0] = CSI.Formats[FormatCombo1.SelectedIndex];
            m_set.videoFormat[1] = CSI.Formats[FormatCombo2.SelectedIndex];

            m_set.UseLSLE[0] = UseLSLECheckBox1.Checked;
            m_set.UseLSLE[1] = UseLSLECheckBox2.Checked;

            m_set.imageFn[0] = ImageFileTextBox1.Text;
            m_set.imageFn[1] = ImageFileTextBox2.Text;

            m_set.VC[0] = VCCombo1.SelectedIndex;
            m_set.VC[1] = VCCombo2.SelectedIndex;

            ParseIntTextBox(PortNumTextBox, "Port", ref m_set.port);

            ParseIntTextBox(HActiveTextBox1, "HActive1", ref m_set.hact[0]);
            ParseIntTextBox(HActiveTextBox2, "HActive2", ref m_set.hact[1]);

            ParseIntTextBox(VActiveTextBox1, "VActive1", ref m_set.vact[0]);
            ParseIntTextBox(VActiveTextBox2, "VActive2", ref m_set.vact[1]);

            ParseIntTextBox(HBlankTextBox1, "HBlank1", ref m_set.hblank[0]);
            ParseIntTextBox(HBlankTextBox2, "HBlank2", ref m_set.hblank[1]);

            ParseIntTextBox(VBlankTextBox1, "VBlank1", ref m_set.vblank[0]);
            ParseIntTextBox(VBlankTextBox2, "VBlank2", ref m_set.vblank[1]);

            m_set.equalLines = EqualLinesCheckBox.Checked;
            m_set.separateFSFE = SeparateFSFECheckBox.Checked;
            m_set.quantLineLength = QuantizeCheckBox.Checked;
            m_set.clockOffDuringBlanking = ClockOffDuringBlankingCheckBox.Checked;
            m_set.loopFrames = LoopCheckBox.Checked;

            m_set.seqFormat = SeqFormatCombo.SelectedIndex;

            // get laen count setting
            // set m_needMeasure if setting has changed
            int laneCnt = LaneCntCombo.SelectedIndex + 1;
            m_needMeasure |= (laneCnt != m_set.laneCnt);
            m_set.laneCnt = laneCnt;

            // get HS frequency setting
            // set m_needMeasure if setting has changed
            float hsFreq = 0.0f;
            ParseFloatTextBox(HSFreqTextBox, "HSFreq", 1e+6f, ref hsFreq);
            m_needMeasure |= (hsFreq != m_set.hsFreq);
            m_set.hsFreq = hsFreq;
        }

        // Event handler for the Connect button
        // Based on current connection state, either connect or disconnect to DPhyGenCtl
        public void ConnectButton_Click(object sender, EventArgs e)
        {
            ClearStatus();

            // not connected, so connect
            if (!m_connected)
            {
                // connect to DPhyGenCtl
                Status("Connecting to DPhyGenCtl...");
                int rc = m_client.Connect("", m_set.port);

                // print error if connect call failed
                if (rc < 0)
                    Status("Can't connect to DPhyGenCtl at port " + m_set.port.ToString());

                // otherwise, connection succeeded
                // update status and control state
                else
                {
                    Status("Connected to DPhyGenCtl at port " + m_set.port.ToString());
                    connectedLabel.Text = "Connected";
                    ConnectButton.Text = "Disconnect";
                    m_connected = true;
                }
            }

            // otherwise, we're connected, so disconnect
            else
            {
                // disconnect from DPhyGenCtl
                m_client.Disconnect(true);

                // update status and control state
                Status("Disconnected from DPhyGenCtl");
                connectedLabel.Text = "Offline";
                ConnectButton.Text = "Connect";
                m_connected = false;
            }
        }

        // send frame
        public void SendFrameButton_Click(object sender, EventArgs e)
        {
            Status("");
            int rc = Prepare();
            if (rc < 0)
            {
                DefaultPE(rc, "initializing DPhyGenCtl");
                return;
            }

            // send frame as a macro
            rc = SendFrame();
            if (rc < 0) 
                DefaultPE(rc, "sending frame");
        }

        // Generic textbox routine to convert return key to tab key
        public void TextBox_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == System.Windows.Forms.Keys.Return)
                this.SelectNextControl((Control)sender, true, true, false, true);
        }

        // Generic textbox routine to process textbox (and other control) contents when it loses focus
        public void TextBox_Leave(object sender, EventArgs e)
        {
            statusLabel.Text = "";
            GetControlData();
        }

        // event handler for image file 2 Browse button
        public void BrowseButton1_Click(object sender, EventArgs e)
        {
            BrowseForImageFile(ImageFileTextBox1);
        }

        // event handler for image file 2 Browse button
        public void BrowseButton2_Click(object sender, EventArgs e)
        {
            BrowseForImageFile(ImageFileTextBox2);
        }

        // Handle MinHBlankButton1 click
        public void MinHBlankButton1_Click(object sender, EventArgs e)
        {
            Status("");
            int rc = ComputeMinHBlank(0);
            if (rc < 0)  {
                DefaultPE(rc, "computing minimum HBlank");
                return;
            }
            m_set.hblank[0] = rc;
            SetControlData();
        }

        // Handle MinHBlankButton2 click
        public void MinHBlankButton2_Click(object sender, EventArgs e)
        {
            Status("");
            int rc = ComputeMinHBlank(1);
            if (rc < 0)
            {
                DefaultPE(rc, "computing minimum HBlank");
                return;
            }
            m_set.hblank[1] = rc;
            SetControlData();
        }

        // Handle SeqFormat selection
        public void SeqFormatCombo_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (m_lock) return;
            GetControlData();
            SetControlData();
        }

        //////////////////////////////////////////////////////////////////////////////////////////
        // File load/save routines
        //////////////////////////////////////////////////////////////////////////////////////////

        public void loadCfgToolStripMenuItem_Click(object sender, EventArgs e)
        {
            OpenFileDialog dlg = new OpenFileDialog();
            dlg.Filter = "CFG files|*.cfg";
            dlg.CheckFileExists = true;
            //            dlg.InitialDirectory = Path.GetDirectoryName();

            Status("");
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                if (!LoadCfg(dlg.FileName))
                    Status("Error loading file " + dlg.FileName);
                else
                    Status("File loaded.");
                SetControlData();
            }
        }

        public void saveCfgToolStripMenuItem_Click(object sender, EventArgs e)
        {
            SaveFileDialog dlg = new SaveFileDialog();
            dlg.Filter = "CFG files|*.cfg";
            //            dlg.InitialDirectory = Path.GetDirectoryName();

            Status("");
            GetControlData();
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                if (!SaveCfg(dlg.FileName))
                    Status("Error saving file " + dlg.FileName);
                else
                    Status("File saved.");
            }

        }

        public bool LoadCfg(string fileName)
        {
            FileStream fs;
            BinaryFormatter formatter = new BinaryFormatter();
            Status("");

            try
            {
                fs = new FileStream(fileName, FileMode.Open, FileAccess.Read);

                int version = (int)formatter.Deserialize(fs);
                m_set = (Settings)formatter.Deserialize(fs);

                fs.Close();
                return true;
            }
            catch
            {
                return false;
            }
        }

        public bool SaveCfg(string fileName)
        {
            FileStream fs;
            BinaryFormatter formatter = new BinaryFormatter();
            Status("");

            try
            {
                fs = new FileStream(fileName, FileMode.OpenOrCreate, FileAccess.Write);

                formatter.Serialize(fs, CONFIG_FILE_VERSION);
                formatter.Serialize(fs, m_set);

                fs.Close();
                return true;
            }
            catch
            {
                return false;
            }
        }

        private void aboutToolStripMenuItem1_Click(object sender, EventArgs e)
        {
            AboutBox dlg = new AboutBox();
            dlg.ShowDialog();
        }

        //////////////////////////////////////////////////////////////////////////////////////////
        // Frame component measurement and parameter computatation routines
        //////////////////////////////////////////////////////////////////////////////////////////

        // This routine measures the number of UI (per lane) required to
        // send a short packet (in this case, Frame Start) in a HS burst.
        // This result includes SOT and EOT for the burst but not clock on/off.
        public int MeasureShortPktUIPerLane()
        {
            int rc;
            string eMsg = "";
            string sMsg = "";

            // set option to turn on the clock before a command is sent to ensure
            // the measurement doesn't include clock-on time
            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_CLOCK_ON_WHEN_SENDING_CMDS,
                                      1, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            // build and measure macro with single (Frame Start) packet
            rc = m_client.RPCCmd(RPCCmds.START_MACRO, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = MIPICmd(RPCDefs.FRAME_START);
            if (rc < 0) return (rc);

            rc = m_client.RPCCmd(RPCCmds.MEASURE_MACRO, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            // Non-negative value returned from MEASURE_MACRO is the number of UI
            // per lane to implement the macro.
            int UI = rc;

            // clear (restore) option to turn on the clock before a command is sent
            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_CLOCK_ON_WHEN_SENDING_CMDS,
                                      0, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            // return number of UI required for short packet
            return (UI);
        }

        public int MeasureClkOnUIPerLane()
        {
            int rc;
            string eMsg = "";
            string sMsg = "";

            // clear option to turn on the clock before a command is sent to ensure
            // the measurement includes clock-on time
            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_CLOCK_ON_WHEN_SENDING_CMDS,
                                      0, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            // build and measure macro with single (Clock On) packet

            rc = m_client.RPCCmd(RPCCmds.START_MACRO, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = MIPICmd(RPCDefs.CLOCK_ON);
            if (rc < 0) return (rc);

            rc = m_client.RPCCmd(RPCCmds.MEASURE_MACRO, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            //            rc = m_client.RPCCmd(RPCCmds.SEND_MACRO, ref eMsg, ref sMsg);
            //            if (rc < 0) return PE(rc, eMsg);

            // Non-negative value returned from MEASURE_MACRO is the number of UI
            // per lane to implement the macro.  Return this value.
            return (rc);
        }

        public int MeasureClkOnOffUIPerLane()
        {
            int rc;
            string eMsg = "";
            string sMsg = "";

            // clear option to turn on the clock before a command is sent to ensure
            // the measurement includes clock-on time
            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_CLOCK_ON_WHEN_SENDING_CMDS,
                                      0, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            // Build and measure macro with (Clock On, Clock Off) packet
            //
            // Note: we can't measure Clock Off as single command in macro because it
            // Clock Off at the start of a macro is treated specially (telling
            // DPhyGenCtl not to turn on the clock the clock at the beginning of 
            // the command).

            rc = m_client.RPCCmd(RPCCmds.START_MACRO, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = MIPICmd(RPCDefs.CLOCK_ON);
            if (rc < 0) return (rc);
            rc = MIPICmd(RPCDefs.CLOCK_OFF);
            if (rc < 0) return (rc);

            rc = m_client.RPCCmd(RPCCmds.MEASURE_MACRO, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            //            rc = m_client.RPCCmd(RPCCmds.SEND_MACRO, ref eMsg, ref sMsg);
            //            if (rc < 0) return PE(rc, eMsg);

            // Non-negative value returned from MEASURE_MACRO is the number of UI
            // per lane to implement the macro.  Return this value.
            return (rc);
        }

        // This routine constructs lines and measures the line length for various
        // parameter combinations.  Ther results are saved in the local variable
        // meas and can be viewed in the debugger at the end of the routine.
        //
        // meas[00]: FS,    ActLines, HBlank
        // meas[01]: Blank, ActLines, HBlank
        // meas[02]: Blank, ActLines, FE, HBlank-
        //
        // meas[03]: FS,    ActLines, ClkOff, HBlank, ClkOn
        // meas[04]: Blank, ActLines, ClkOff, HBlank, ClkOn
        // meas[05]: Blank, ActLines, FE, ClkOff, HBlank-, ClkOn
        // 
        // meas[06]: FS,    LS, ActLines, LE, HBlank
        // meas[07]: Blank, LS, ActLines, LE, HBlank
        // meas[08]: Blank, LS, ActLines, LE, FE, HBlank-
        //
        // meas[09]: FS,    LS, ActLines, LE, ClkOff, HBlank, ClkOn
        // meas[10]: Blank, LS, ActLines, LE, ClkOff, HBlank, ClkOn
        // meas[11]: Blank, LS, ActLines, LE, FE, ClkOff, HBlank-, ClkOn
        //
        // meas[12]: ActLines, HBlank
        // meas[13]: ActLines, HBlank
        // meas[14]: ActLines, HBlank
        //
        // meas[15]: ActLines, ClkOff, HBlank, ClkOn
        // meas[16]: ActLines, ClkOff, HBlank, ClkOn
        // meas[17]: ActLines, ClkOff, HBlank, ClkOn
        public int MeasureLineTest(int frameInx)
        {
            int rc;
            string eMsg = "";
            string sMsg = "";

            // request that DPhyGenCtl load video frame into frame buffer 0
            rc = m_client.RPCCmd(RPCCmds.LOAD_FRAME, 0, m_set.videoFormat[frameInx],
                                      m_set.imageFn[frameInx], ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            // Measure the line lengths (in UI) for various combinations
            // of options: (ClockOffDuringBlanking, UseLSLE, separate FSFE)
            // Results are stored in meas array (review in debugger)
            int[] meas = new int[18];
            for (int i = 0; i < 18; i++)
            {
                bool fs = ((i % 3) == 0);
                bool fe = ((i % 3) == 2);
                m_set.clockOffDuringBlanking = ((i % 6) >= 3);
                m_set.UseLSLE[0] = ((i >= 6) && (i < 12));
                m_set.separateFSFE = ((i >= 12) && (i < 18));

                rc = m_client.RPCCmd(RPCCmds.START_MACRO, ref eMsg, ref sMsg);
                if (rc < 0) return PE(rc, eMsg);

                rc = AddActiveLine(fs, fe, 0, 0, false);
                if (rc < 0) return (rc);

                rc = m_client.RPCCmd(RPCCmds.MEASURE_MACRO, ref eMsg, ref sMsg);
                if (rc < 0) return PE(rc, eMsg);

                meas[i] = rc;
            }

            // deallocate frame buffer in DPhyGenCtl
            rc = m_client.RPCCmd(RPCCmds.DEALLOC_FRAME, 0, 0, "", ref eMsg, ref sMsg);
            if (rc < 0) return (rc);

            return (0);
        }

        // Compute bytes per line per lane given the number of bits per line
        // If the value is not integral, quantize or return error depending on setting.
        public int ComputeBytesPerLinePerLane(int bitsPerLine)
        {
            int bytesPerLinePerLane = bitsPerLine / (8 * m_set.laneCnt);

            int rem = bitsPerLine % (8 * m_set.laneCnt);
            if (rem != 0)
            {
                // if allowed, quantize line length to next byte boundary
                if (m_set.quantLineLength)
                    bytesPerLinePerLane++;

                // otherwise, display and return error
                else
                {
                    Status("Error: total line length must be an integral number of bytes per lane.");
                    return (-1);
                }
            }

            // return result
            return (bytesPerLinePerLane);
        }

        // Compute the minimum horizontal blanking requirements
        // (assumes DeriveSettings has been called already)
        public int ComputeMinHBlank(int frameInx)
        {
            int rc;
            string eMsg = "";
            string sMsg = "";

            // Set up DPhyGenCtl global options/parameters
            rc = Prepare();
            if (rc < 0) return (rc);

            // build and measure macro with single (Frame Start) packet
            rc = m_client.RPCCmd(RPCCmds.START_MACRO, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            // send active frame line
            //
            // (turn both FS and FE on to account for these periods in each line, even though
            // FS is only normally sent on the first active line and FE is only normally 
            // sent on the last active line)
            rc = AddActiveLine(!m_set.separateFSFE, !m_set.separateFSFE, frameInx, 0, true);
            if (rc < 0) return (rc);

            rc = m_client.RPCCmd(RPCCmds.MEASURE_MACRO, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            // Non-negative value returned from MEASURE_MACRO is the number of UI to implement the line
            int hblankUI = rc;

            // convert to pixels across lanes (round up)
            int minHBlankPix = ((hblankUI * m_set.laneCnt + m_set.bitsPerPix[frameInx] - 1) / m_set.bitsPerPix[frameInx]);
            return (minHBlankPix);
        }

        // Call this routine whenever m_set user-fields have changed to
        // derive/compute/measure other parameters for frame construction.
        public int DeriveSettings()
        {
            int rc;

            Status("Deriving video construction parameters...");

            // compute derived ine/frame parameters and metrics
            m_set.frameCnt = (m_set.seqFormat == Settings.SEQ_SINGLE_FRAME) ? 1 : 2;
            for (int i = 0; i < m_set.frameCnt; i++)
            {
                // get RPC command for this video type
                m_set.rpcVideoCmd[i] = CSI.CSIVideoCmdToRPCCmd(m_set.videoFormat[i]);

                // get the number of bits per pixel for this video type
                m_set.bitsPerPix[i] = CSI.CSIVideoCmdBitsPerPixel(m_set.videoFormat[i]);

                // compute the total number of bits in a video line
                m_set.bitsPerLine[i] = (m_set.hact[i] + m_set.hblank[i]) * m_set.bitsPerPix[i];

                // compute the total number of bytes per lane in a video line
                rc = ComputeBytesPerLinePerLane(m_set.bitsPerLine[i]);

                // return if bytesPerLinePerLane is not an integral value
                // (and quantization was not allowed)
                if (rc < 0) return (-1);

                m_set.bytesPerLinePerLane[i] = rc;

                // compute number of active bytes in a video line
                int activeLineLenInBits = m_set.bitsPerPix[i] * m_set.hact[i];
                if ((activeLineLenInBits & 7) != 0)
                {
                    Status("Error: active line length is not an integral number of bytes");
                    return (-1);
                }
                m_set.activeLineLenInBytes[i] = activeLineLenInBits >> 3;
            }

            // Handle multi-frame case where equalLines setting is true
            // This means, that the shorter line is padded with additional blanking to make equal
            if ((m_set.frameCnt > 1) && m_set.equalLines)
            {
                m_set.bytesPerLinePerLane[0] = Math.Max(m_set.bytesPerLinePerLane[0], m_set.bytesPerLinePerLane[1]);
                m_set.bytesPerLinePerLane[1] = m_set.bytesPerLinePerLane[0];
            }

            // If settings have changed that require new measures of Clock On, Clock Off, 
            // and short packet time, perform the measurements and record the results.
            if (m_needMeasure)
            {
                m_set.shortPktUIPerLane = MeasureShortPktUIPerLane();
                m_set.clkOnUIPerLane = MeasureClkOnUIPerLane();
                m_set.clkOffUIPerLane = MeasureClkOnOffUIPerLane() - m_set.clkOnUIPerLane;
                m_needMeasure = false;
            }

            Status("Parameter derivation complete.");

            return (0);
        }

        //////////////////////////////////////////////////////////////////////////////////////////
        // Frame construction routines
        //////////////////////////////////////////////////////////////////////////////////////////

        public int MIPICmd(int rpcCmd)
        {
            return (MIPICmd(rpcCmd, 0, 0, 0, 0, ""));
        }

        public int MIPICmd(int rpcCmd, int VC)
        {
            return (MIPICmd(rpcCmd, VC, 0, 0, 0, ""));
        }

        public int MIPICmd(int rpcCmd, int VC, int arg1, int arg2, int arg3, string fn)
        {
            string eMsg = "";
            string sMsg = "";
            int rc = (m_client.MIPICmd(rpcCmd, 0, false,
                RPCDefs.DT_HS, VC, arg1, arg2, arg3, fn, null,
                ref eMsg, ref sMsg));
            if (rc < 0) return PE(rc, eMsg);
            return (0);
        }

        public int InitDPhyGenCtl()
        {
            int rc;
            string eMsg = "";
            string sMsg = "";

            Status("Initializing DPhyGenCtl parameters...");

            rc = m_client.RPCCmd(RPCCmds.SET_MIPI_STANDARD, RPCDefs.STD_CSI, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.START_EDIT_CONFIG, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.SET_HS_BIT_RATE, m_set.hsFreq, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.SET_LANE_CNT, m_set.laneCnt, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.END_EDIT_CONFIG, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_SEND_SINGLE_PKT_PER_HS_BURST,
                                      1, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_ENABLE_VIDEO_MODE_IN_MACROS,
                                      0, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_CLOCK_ON_WHEN_SENDING_CMDS,
                                      0, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_CLOCK_OFF_AFTER_HS_BURST,
                                      0, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_LOOP_COMMANDS,
                                      m_set.loopFrames ? 1 : 0, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            rc = m_client.RPCCmd(RPCCmds.SET_OPTION, RPCDefs.OPT_ENABLE_CMD_INSERTION,
                                      0, ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            Status("DPhyGenCtl initialization complete.");

            return (0);
        }

        public int AddVBlanking(int blankBytesPerLane)
        {
            int rc;

            if (blankBytesPerLane == 0)
                return(0);

            // mark the current byte position in the macro
            rc = MIPICmd(RPCDefs.MARK_ZERO_POS);
            if (rc < 0) return rc;

            // turn clock off for blanking, if requested
            if (m_set.clockOffDuringBlanking)
            {
                rc = MIPICmd(RPCDefs.CLOCK_OFF);
                if (rc < 0) return rc;
            }

            // compute blanking time
            // Subtract the number of UI required to turn on the clock, if it is off
            int toPosUI = 8 * blankBytesPerLane;
            if (m_set.clockOffDuringBlanking)
                toPosUI -= m_set.clkOnUIPerLane;

            // add LP11 blanking
            rc = MIPICmd(RPCDefs.LP_DELAY_TO_POS_UI, 0, toPosUI, 0x3ff, 0, "");
            if (rc < 0) return rc;

            // turn clock on, if requested
            if (m_set.clockOffDuringBlanking)
            {
                rc = MIPICmd(RPCDefs.CLOCK_ON);
                if (rc < 0) return rc;
            }

            return (0);
        }

        // Adds an active line for the given frame to the current macro (assumes START_MACRO has been called)
        //
        // Set addFrameStart to add a Frame Start packet at the start of the line.
        // Otherwise, a period of LP11 blanking will be output at the start of the line instead.
        // Set addFrameEnd to add a FrameEnd packet after the active video pacekt.
        // Set the lineInx to the active line number (zero-based)
        //
        // Set testMeasure = true, to construct a minimum length video line for measurement.
        //
        // This routine assumes the clock is running on entry, otherwise, when the FRAME_START is sent
        // the clock will first automatically be turned on, resulting in different timing than
        // that expected.
        //
        // This routine ensures the clock running on exit.
        public int AddActiveLine(bool addFrameStart, bool addFrameEnd,
                                 int frameInx, int lineInx, bool testMeasure)
        {
            int rc;

            // mark the current byte position in the macro
            rc = MIPICmd(RPCDefs.MARK_ZERO_POS);
            if (rc < 0) return rc;

            // add FrameStart, if requested
            if (addFrameStart)
            {
                rc = MIPICmd(RPCDefs.FRAME_START, m_set.VC[frameInx]);
                if (rc < 0) return rc;
            }

            // otherwise, add LPDelay equal to the FrameStart time 
            // (if user has not asked for the FrameStart to be separate from first line)
            else if (!m_set.separateFSFE)
            {
                rc = MIPICmd(RPCDefs.LP_DELAY_TO_POS_UI, 0, m_set.shortPktUIPerLane, 0x3ff, 0, "");
                if (rc < 0) return rc;
            }

            // add LineStart, if requested
            if (m_set.UseLSLE[frameInx])
            {
                rc = MIPICmd(RPCDefs.LINE_START, m_set.VC[frameInx]);
                if (rc < 0) return rc;
            }

            // If we're constructing this video line for measurement,
            // use dummy null packet as place-holder packet
            if (testMeasure)
            {
                rc = MIPICmd(RPCDefs.CSI_NULL_PKT, 0);
                if (rc < 0) return rc;
            }

            // Add video packet
            else
            {
#if true
                // Use USERFRAME<n> as special file name to access frame buffer data 
                // loaded by LOAD_FRAME.  When a frame buffer is used, the pixel stream
                // commands use arg1 = byte offset in frame and arg2 = byte length of packet
                int frameOffset = lineInx * m_set.activeLineLenInBytes[frameInx];
                string frameBufName = "USERFRAME" + frameInx.ToString();
                rc = MIPICmd(m_set.rpcVideoCmd[frameInx], m_set.VC[frameInx], frameOffset,
                             m_set.activeLineLenInBytes[frameInx], 0, frameBufName);
                if (rc < 0) return rc;

#else
                // debug: send video data directly
                string eMsg = "";
                string sMsg = "";
                byte[] lineBuf = new byte[m_set.activeLineLenInBytes[frameInx]];
                for (int i = 0; i < lineBuf.Length; i++)
                    lineBuf[i] = (byte)(i + 1);
                rc = (m_client.MIPICmd(m_set.rpcVideoCmd[frameInx], 0, false,
                    RPCDefs.DT_HS, m_set.VC[frameInx], 0, 0, 0, "", lineBuf,
                    ref eMsg, ref sMsg));
#endif
            }

            // add LineEnd, if requested
            if (m_set.UseLSLE[frameInx])
            {
                rc = MIPICmd(RPCDefs.LINE_END, m_set.VC[frameInx]);
                if (rc < 0) return rc;
            }

            // add FrameEnd, if requested
            if (addFrameEnd)
            {
                rc = MIPICmd(RPCDefs.FRAME_END, m_set.VC[frameInx]);
                if (rc < 0) return rc;
            }

            // turn clock off for blanking, if requested
            if (m_set.clockOffDuringBlanking)
            {
                rc = MIPICmd(RPCDefs.CLOCK_OFF);
                if (rc < 0) return rc;
            }

            // add HFP blanking
            //
            // If we are measuring this line to determine the minimum HFP blanking requirement
            // then don't add any blanking here (this will yield the minimum length line)
            if (!testMeasure)
            {
                // compute blanking time
                // Total line length is bytesPerLinePerLane
                // Subtract the number of UI required to turn on the clock, if it is off
                int toPosUI = 8 * m_set.bytesPerLinePerLane[frameInx];
                if (m_set.clockOffDuringBlanking)
                    toPosUI -= m_set.clkOnUIPerLane;

                // add LP11 blanking
                rc = MIPICmd(RPCDefs.LP_DELAY_TO_POS_UI, 0, toPosUI, 0x3ff, 0, "");
                if (rc < 0) return rc;
            }

            // turn clock on, if requested
            if (m_set.clockOffDuringBlanking)
            {
                rc = MIPICmd(RPCDefs.CLOCK_ON);
                if (rc < 0) return rc;
            }

            return (0);
        }

        // Prepare to send commands to DPhyGenCtl
        //
        // 1) Read user settings from controls
        // 2) Checks that we have conntection to DPhyGenCtl
        // 3) Initializes global settings/options in DPhyGenCtl
        // 4) Compute frame-building parameters from user-settings
        public int Prepare()
        {
            int rc;
            Status("");

            // Read user settings
            GetControlData();

            // Check/ensure we have connection to DPhyGenCtl
            if (!m_connected)
                ConnectButton_Click(null, null);
            if (!m_connected) return (-1);

            // initialize DPhyGenCtl settings
            rc = InitDPhyGenCtl();
            if (rc < 0) return (rc);

            // derive frame-building parameters from user-settings
            rc = DeriveSettings();
            if (rc < 0) return (rc);

            return (0);
        }

        public int AddActiveLines(int frameInx, bool noFS, bool noFE)
        {
            int rc; 
            int pct = -1;
            int actLines = m_set.vact[frameInx];
            for (int lineInx = 0; lineInx < actLines; lineInx++)
            {
                // display status
                int curPct = (100 * lineInx) / actLines;
                if (curPct > pct)
                {
                    Status(string.Format("Building frame {0} -- {1}% complete", frameInx+1, pct));
                    pct = curPct;
                }

                // set flags to include FS and FE in current line
                bool fs = (lineInx == 0) & !m_set.separateFSFE & !noFS;
                bool fe = (lineInx == m_set.vact[frameInx] - 1) & !m_set.separateFSFE & !noFE;

                // send active video line
                rc = AddActiveLine(fs, fe, frameInx, lineInx, false);
                if (rc < 0) return (rc);
            }

            return (0);
        }

        // Interleaves lines in alternating pattern until all lines from shorter frame
        // are gone, then send remaining lines of longer frame
        public int AddInterleavedActiveLinesOneForOne()
        {
            int rc;
            int pct = -1;
            int act1 = m_set.vact[0];
            int act2 = m_set.vact[1];
            int maxAct = Math.Max(act1, act2);
            for (int lineInx = 0; lineInx < maxAct; lineInx++)
            {
                // display status
                int curPct = (100 * lineInx) / maxAct;
                if (curPct > pct)
                {
                    Status("Building frame -- " + curPct.ToString() + "% complete");
                    pct = curPct;
                }

                // add line from frame 1, then line from frame 2
                for (int frameInx = 0; frameInx < 2; frameInx++)
                {
                    // set flags to include FS and FE in current line
                    bool fs = (lineInx == 0) & !m_set.separateFSFE;
                    bool fe = (lineInx == m_set.vact[frameInx] - 1) & !m_set.separateFSFE;

                    // if frame 1 VC is same as frame 2 VC, skip FS/FE for frame 2
                    if ((frameInx == 1) & (m_set.VC[0] == m_set.VC[1]))
                    {
                        fs = false;
                        fe = false;
                    }

                    // send active video line
                    if (lineInx < m_set.vact[frameInx])
                    {
                        rc = AddActiveLine(fs, fe, frameInx, lineInx, false);
                        if (rc < 0) return (rc);
                    }
                }
            }

            return (0);
        }

        // Interleaves lines, distributing throughout frame as best we can
        public int AddInterleavedActiveLinesDistrib()
        {
            int rc;
            int pct = -1;
            int act1 = m_set.vact[0];
            int act2 = m_set.vact[1];
            int totAct = act1 + act2;
            int lineInx1 = 0;
            int lineInx2 = 0;
            int lineInx = 0;
            int frameInx = 0;
            bool sameVC = (m_set.VC[0] == m_set.VC[1]);
            for (int totLineInx = 0; totLineInx < totAct; totLineInx++)
            {
                // display status
                int curPct = (100 * totLineInx) / totAct;
                if (curPct > pct)
                {
                    Status("Building frame -- " + curPct.ToString() + "% complete");
                    pct = curPct;
                }

                // if frame 1 is complete, process line from frame 2
                if (lineInx1 == act1)
                    frameInx = 1;

                // if frame 2 is complete, process line from frame 1
                else if (lineInx2 == act2)
                    frameInx = 0;

                // otherwise, decide which frame has made less progress
                else
                {
                    double pct1 = ((double)lineInx1 / act1);
                    double pct2 = ((double)lineInx2 / act2);
                    frameInx = (pct1 <= pct2) ? 0 : 1;
                }

                // get next line to output for this frame
                if (frameInx == 0)
                    lineInx = lineInx1++;
                else
                    lineInx = lineInx2++;

                // set flags to include FS and FE in current line
                bool fs = (lineInx == 0) & !m_set.separateFSFE;
                bool fe = (lineInx == m_set.vact[frameInx] - 1) & !m_set.separateFSFE;

                // if frame 1 VC is same as frame 2 VC, skip second FS and first FE
                if (sameVC && (totLineInx != 0))
                    fs = false;
                if (sameVC && (totLineInx != (totAct - 1)))
                    fe = false;

                // send active video line
                if (lineInx < m_set.vact[frameInx])
                {
                    rc = AddActiveLine(fs, fe, frameInx, lineInx, false);
                    if (rc < 0) return (rc);
                }
            }

            return (0);
        }

        // add frame start or frame end sequence for concentric frames
        public int AddConcentricFSFE(bool FS)
        {
            int rc = 0;
            int cmd = FS ? RPCDefs.FRAME_START : RPCDefs.FRAME_END;

            // send Frame Start if FS and FE are requested to be on separate lines
            if (m_set.separateFSFE)
            {
                // send frame start/end for frame 1
                rc = MIPICmd(cmd, m_set.VC[0]);
                if (rc < 0) return rc;

                // send frame start/end for frame 2 if different VC
                if (m_set.VC[1] != m_set.VC[0])
                {
                    rc = MIPICmd(cmd, m_set.VC[1]);
                    if (rc < 0) return rc;
                }
            }

            return (rc);
        }

        public int AddSingleFrame(int frameInx)
        {
            int rc;

            

            // send Frame Start if FS and FE are requested to be on separate lines
            if (m_set.separateFSFE)
            {
                rc = MIPICmd(RPCDefs.FRAME_START, m_set.VC[frameInx]);
                if (rc < 0) return rc;
            }

            // add active lines
            rc = AddActiveLines(frameInx, false, false);
            if (rc < 0) return (rc);

            // send Frame End if FS and FE are requested to be on separate lines
            if (m_set.separateFSFE)
            {
                rc = MIPICmd(RPCDefs.FRAME_END, m_set.VC[frameInx]);
                if (rc < 0) return rc;
            }

            // send LP11 for vblank lines 
            // (turn clock off/on if requested)
            rc = AddVBlanking(m_set.vblank[frameInx] * m_set.bytesPerLinePerLane[frameInx]);
            return (rc);
        }

        // Add concentric sequential frames to macro
        //
        // If VC is the same for frame 1 and frame 2, only one FS and FE are sent
        //
        // VC1 == VC2
        //     FS Frame 1, Frame 2 FE, VBlank1, VBlank2
        //     If SeparateFSFE, 
        //         FS and FE are on separate lines.
        //     If !SeparateFSFE,
        //         FS will be on frame1, line1 
        //         FE will be on frame2 last line
        //
        // VC1 != VC2
        //     If SeparateFSFE, 
        //         FS1, FS2, Frame 1, Frame 2, FE1, FE2, VBlank1, VBlank2
        //     If !SeparateFSFE,
        //         <same as sequenctial frames>
        public int AddConcentricSequentialFrames()
        {
            int rc;

            // add frame start sequence if requested on separate line
            rc = AddConcentricFSFE(true);
            if (rc < 0) return (rc);

            // add active lines for frame 1
            bool sameVC = (m_set.VC[0] == m_set.VC[1]);
            rc = AddActiveLines(0, false, sameVC);
            if (rc < 0) return (rc);

            // add active lines for frame 2
            rc = AddActiveLines(1, sameVC, false);
            if (rc < 0) return (rc);

            // add frame end sequence if requested on separate line
            rc = AddConcentricFSFE(false);
            if (rc < 0) return (rc);

            // send LP11 for combined vblank lines 
            // (turn clock off/on if requested)
            rc = AddVBlanking(m_set.vblank[0] * m_set.bytesPerLinePerLane[0] +
                         m_set.vblank[1] * m_set.bytesPerLinePerLane[1]);
            return (rc);
        }

        // Add concentric interleaved frames to macro
        //
        // If VC is the same for frame 1 and frame 2, only first FS and last FE are sent
        //
        // For frames of equal number of lines:
        // separateFSFE:
        //     FS1, FS2 
        //     Frame 1, Line1, Frame 2 Line1,
        //     Frame 1, Line2, Frame 2 Line2,
        //     ...
        //     FE1, FE2
        //     VBlank1, VBlank2
        //
        // !separateFSFE:
        //     FS1, Frame 1, Line1, FS2, Frame 2 Line1,
        //     Frame 1, Line2, Frame 2 Line2,
        //     ...
        //     Frame 1 last line, FE1, Frame2 last line, FE2
        //     VBlank1, VBlank2
        //
        // For frames of inequal number of lines, interleaved lines will be distributed
        // as evenly as possible throughout frame.
        public int AddConcentricInterleavedFrames()
        {
            int rc; 

            // add frame start sequence if requested on separate line
            rc = AddConcentricFSFE(true);
            if (rc < 0) return (rc);

            // add interleaved active lines from frame 1 and frame 2
            rc = AddInterleavedActiveLinesDistrib();
            if (rc < 0) return (rc);

            // add frame end sequence if requested on separate line
            rc = AddConcentricFSFE(false);
            if (rc < 0) return (rc);

            // send LP11 for combined vblank lines 
            // (turn clock off/on if requested)
            rc = AddVBlanking(m_set.vblank[0] * m_set.bytesPerLinePerLane[0] +
                         m_set.vblank[1] * m_set.bytesPerLinePerLane[1]);
            return (rc);
        }

        public int SendFrame()
        {
            int frameInx = 0;
            int rc = 0;
            string eMsg = "";
            string sMsg = "";

            Status("");

            // Load image frame(s) into DPhyGenCtl frame buffer(s)
            //
            // In case we are talking to DPhyGenCtlForP338, make sure we set HActive and VActive 
            // in DPhyGenCtl to prevent resizing (if image is correct dimensions) or resize to 
            // the correct dimensions otherwise.  This is not required for DPhyGenCtl which does
            // not resize.
            for (frameInx = 0; frameInx < m_set.frameCnt; frameInx++)
            {
                // set active frame dimensions in timing setting
                rc = m_client.RPCCmd(RPCCmds.SET_TIMING_HACTIVE, m_set.hact[frameInx], ref eMsg, ref sMsg);
                if (rc < 0) return (PE(rc, eMsg));
                rc = m_client.RPCCmd(RPCCmds.SET_TIMING_VACTIVE, m_set.vact[frameInx], ref eMsg, ref sMsg);
                if (rc < 0) return (PE(rc, eMsg));

                // load the image file
                rc = m_client.RPCCmd(RPCCmds.LOAD_FRAME, frameInx, m_set.videoFormat[frameInx],
                                          m_set.imageFn[frameInx], ref eMsg, ref sMsg);
                if (rc < 0) return PE(rc, eMsg);
            }

            // start frame macro
            rc = m_client.RPCCmd(RPCCmds.START_MACRO,
                ref eMsg, ref sMsg);

            // turn clock on
            rc = MIPICmd(RPCDefs.CLOCK_ON);
            if (rc < 0) return (rc);

            // Case1: add single frame to macro
            if (m_set.seqFormat == Settings.SEQ_SINGLE_FRAME)
            {
                rc = AddSingleFrame(0);
                if (rc < 0) return (rc);
            }

            // Case2: add consecutive frames to macro (all of frame 1 then all of frame 2)
            else if (m_set.seqFormat == Settings.SEQ_SEQUENTIAL_FRAMES)
            {
                for (frameInx = 0; frameInx < 2; frameInx++)
                {
                    rc = AddSingleFrame(frameInx);
                    if (rc < 0) return (rc);
                }
            }

            // Case3: add concentric sequential frames to macro
            else if (m_set.seqFormat == Settings.SEQ_CONCENTRIC_SEQUENTIAL_FRAMES)
            {
                rc = AddConcentricSequentialFrames();
                if (rc < 0) return (rc);
            }

            // Case4: add concentric interleaved frames to macro
            else if (m_set.seqFormat == Settings.SEQ_CONCENTRIC_INTERLEAVED_FRAMES)
            {
                rc = AddConcentricInterleavedFrames();
                if (rc < 0) return (rc);
            }

            // send frame macro
            Status("Sending macro...");
            rc = m_client.RPCCmd(RPCCmds.SEND_MACRO,
                ref eMsg, ref sMsg);
            if (rc < 0) return PE(rc, eMsg);

            // deallocate frame buffers
            for (frameInx = 0; frameInx < m_set.frameCnt; frameInx++)
            {
                rc = m_client.RPCCmd(RPCCmds.DEALLOC_FRAME, frameInx,
                    ref eMsg, ref sMsg);
                if (rc < 0) return PE(rc, eMsg);
            }

            Status("Macro sent.");

            return (0);
        }



    }
}
